import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import root_scalar

# Define the loss function
def loss_fn(x):
    return np.log(1 + np.exp(-64 * x))

# Function to find x from a known loss
def find_x_for_loss(target_loss, x_range=(-1, 1)):
    def f(x):
        return loss_fn(x) - target_loss
    result = root_scalar(f, bracket=x_range, method='brentq')
    return result.root if result.converged else None

# Loss values are taken from output of function running 1D Regression
loss_values= [
0.20868271653056145,
0.009245626630017068,
0.05630753545463085
]
labels = ['Layer 1', 'Layer 2', 'Layer 3']

# Get x-values for those loss values
x_points = [find_x_for_loss(lv) for lv in loss_values]

# Generate smooth curve
x_curve = np.linspace(-1, 1, 1000)
y_curve = loss_fn(x_curve)

# Plot
plt.figure(figsize=(8, 5))
plt.plot(x_curve, y_curve, label="loss(x)", linewidth=2,color='b')
plt.scatter(x_points, loss_values, color='red', zorder=5)

# Annotate each point
for x, y, label in zip(x_points, loss_values, labels):
    plt.annotate(f"{label}", (x, y), textcoords="offset points", xytext=(10,5), ha='left', fontsize=10)
plt.xlim(-0.025,0.1)
plt.ylim(-0.5,2)
plt.xlabel(r"$\delta = g_{pos} - g_{neg}$")
plt.ylabel(r"$\log\left(1 + \theta e^{-\delta}\right)$")
plt.title("Loss Curve with Layer Markers")
plt.tight_layout()
plt.show()
